#!/usr/bin/env bash

# Function to get domains
get_domains() {
    # Get the input domain from the argument
    local domain=$1

    # Create the SOAP request body with the input domain
    body="<?xml version=\"1.0\" encoding=\"utf-8\"?>
    <soap:Envelope xmlns:exm=\"http://schemas.microsoft.com/exchange/services/2006/messages\" 
        xmlns:ext=\"http://schemas.microsoft.com/exchange/services/2006/types\" 
        xmlns:a=\"http://www.w3.org/2005/08/addressing\" 
        xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\" 
        xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\">
    <soap:Header>
        <a:RequestedServerVersion>Exchange2010</a:RequestedServerVersion>
        <a:MessageID>urn:uuid:6389558d-9e05-465e-ade9-aae14c4bcd10</a:MessageID>
        <a:Action soap:mustUnderstand=\"1\">http://schemas.microsoft.com/exchange/2010/Autodiscover/Autodiscover/GetFederationInformation</a:Action>
        <a:To soap:mustUnderstand=\"1\">https://autodiscover.byfcxu-dom.extest.microsoft.com/autodiscover/autodiscover.svc</a:To>
        <a:ReplyTo>
        <a:Address>http://www.w3.org/2005/08/addressing/anonymous</a:Address>
        </a:ReplyTo>
    </soap:Header>
    <soap:Body>
        <GetFederationInformationRequestMessage xmlns=\"http://schemas.microsoft.com/exchange/2010/Autodiscover\">
        <Request>
            <Domain>${domain}</Domain>
        </Request>
        </GetFederationInformationRequestMessage>
    </soap:Body>
    </soap:Envelope>"

    # Perform the HTTP POST request using curl and store the response
    response=$(curl -s -X POST -H "Content-type: text/xml; charset=utf-8" -H "User-agent: AutodiscoverClient" -d "$body" "https://autodiscover-s.outlook.com/autodiscover/autodiscover.svc")

    # Check if the response is empty, indicating a failed request
    if [[ -z "$response" ]]; then
        echo "[-] Unable to execute request. Wrong domain?"
        exit 1
    fi

    # Parse the XML response to extract domain names using xmllint
    domains=$(echo "$response" | xmllint --xpath '//*[local-name()="Domain"]/text()' -)

    # Check if no domains were found in the response
    if [[ -z "$domains" ]]; then
        echo "[-] No domains found."
        exit 1
    fi

    # Print the found domains
    echo -e "\n[+] Domains found:"
    echo "$domains" | tr ' ' '\n'

    # Extract the tenant name from the found domains
    tenant=$(echo "$domains" | tr ' ' '\n' | grep "onmicrosoft.com" | head -n 1 | cut -d'.' -f1)

    # Check if no tenant was found
    if [[ -z "$tenant" ]]; then
        echo "[-] No tenant found."
        exit 1
    fi

    # Print the found tenant name
    echo -e "\n[+] Tenant found: \n${tenant}"

    # Call the check_mdi function with the tenant name
    check_mdi "$tenant"
}

# Function to identify MDI usage
check_mdi() {
    # Append the MDI domain suffix to the tenant name
    tenant="$1.atp.azure.com"

    # Use dig to check if the MDI instance exists for the tenant domain
    if dig "$tenant" +short; then
        # Print a positive message if the MDI instance was found
        echo -e "\n[+] An MDI instance was found for ${tenant}!\n"
    else
        # Print a negative message if no MDI instance was found
        echo -e "\n[-] No MDI instance was found for ${tenant}\n"
    fi
}

# Main script entry point
# Check if the correct number of arguments is provided and if the first argument is -d
if [[ $# -ne 2 || $1 != "-d" ]]; then
    # Print the usage information if the arguments are incorrect
    echo "Usage: $0 -d <domain>"
    exit 1
fi

# Get the domain argument from the input
domain=$2

# Call the get_domains function with the provided domain
get_domains "$domain"
