#!/usr/bin/env bash

if [ "$#" -ne 1 ]; then
    echo "Converts Nuclei ssl-dns-names scan output to hosts file format"
    echo "Usage: $0 /path/to/file"
    exit 1
fi

# Pass the file content to the cut command.
# Cut using a space as delimeter from the 4th field to the end of line.
# A backslash at the end of a line continues the command on the next line.
cat "$1" | cut -d ' ' -f 4- | \

# Multi-part sed commands separated by semi-colon:
# Removes the string :443 from the input.
# Removes all occurrences of the [ character from the input. The g at the end means “global,” so it applies the substitution to all matches in each line.
# Removes all occurrences of the ] character from the input (global).
# Removes all occurrences of the double quote (") character from the input (global).
# Replaces all occurrences of the comma (,) character with a space ( ) (global).
# Removes any space followed by *. and any sequence of non-space characters. This typically removes wildcard subdomain entries like  *.example.com (global).
sed 's/:443//;s/\[//g;s/\]//g;s/"//g;s/,/ /g;s/ \*\.[^ ]*//g' | \

# Ensure each line is unique based on the 2nd column.
# Sort unique (-u) based on data in the 2nd column (-k2,2). After -k the values are start and stop fields.
sort -u -k2 | \

awk '{
    # Initialize new_line as an empty string
    new_line = ""
    # Iterate over all fields in the current record
    for (i = 1; i <= NF; i++) {
	# If the field does not contain an asterisk, add it to the new line
	if ($i !~ /\*/) {
	    new_line = new_line $i " "
	}
    }
    # Trim trailing space
    sub(/[ \t]+$/, "", new_line)
    # Split new_line into an array called fields using space as the delimiter
    # Print the new line only if it contains more than one column
    if (split(new_line, fields, " ") > 1) {
	print new_line
    }
}'
